/**
* \file: Session.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPLay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef CARPLAY_SESSION_H
#define CARPLAY_SESSION_H

#include <memory>
#include "AirPlayHeaders.h"
#include "IInputAdapter.h"
#include "IControlAdapter.h"
#include "utils/Referencable.h"
#include "input/InputChannel.h"
#include "audio/AudioChannel.h"
#include "Server.h"
#include <pthread_adit.h>
#include <mutex>              // std::mutex, std::unique_lock
#include <condition_variable> // std::condition_variable
#include <sys/eventfd.h>


namespace adit { namespace carplay
{

enum NightMode
{
    NightMode_NotSupported = -1,
    NightMode_Disabled = 0,
    NightMode_Enabled = 1
};


class CARPLAY_HIDDEN Session : public IControlReceiver,
    public Referencable<Session, AirPlayReceiverSessionRef>
{
public:
    Session();
    ~Session();

    bool Start(Server& inServer, IDynamicConfiguration& inConfig);
    void Stop(bool inKeepObjectAlive);

    AudioFormats GetAudioFormats(AudioChannelType inChannelType,AudioType inStreamType);
    AudioFormats GetCompatibilityAudioFormats(bool inDirection, AudioChannelType inChannelType, AudioType inStreamType);
    inline std::list<std::string> GetBluetoothIDs() { return bluetoothIDs; };
    inline NightMode GetNightMode() { return nightMode; };
    inline bool GetLimitedUI() { return limitedUI; }
    inline bool GetUpdateVehicleInfo() { return updateVehicleInfo;}
    inline ETCInfo GetVehicleETCSupport() { return etcSupport;}
    inline NADInfo GetVehicleNADSupport() { return nadSupport;}
    inline void SetSupportLimitedUI(bool isSupported) { supportLimitedUI = isSupported; }

    void RampVolume(double inFinalVolume, double inDuration);

    void AudioPrepare(AudioChannelType inChannel, std::string inAudioType);
    void AudioStop(AudioChannelType inChannel);
    void VideoPlaybackStarted();
    struct sockaddr const * getIPAddress();

    inline IConfiguration* GetConfig() { return config; }
    inline void SetAudioChannel(AudioChannel* inChannel) { audioChannel = inChannel; }

    // interface implementations
private:
    // AirPlayReceiverSession delegates
    static OSStatus HandleSessionInitialized(AirPlayReceiverSessionRef inSession, void *inContext);
    static void HandleSessionStarted(AirPlayReceiverSessionRef inSession, void *inContext);
    static void HandleSessionFinalized(AirPlayReceiverSessionRef inSession, void* inContext);
    static void HandleModesChanged(AirPlayReceiverSessionRef inSession,
            const AirPlayModeState* inState, void* inContext);
    static void HandleRequestUI(AirPlayReceiverSessionRef inSession, CFStringRef inURL, void* inContext);
    static OSStatus HandleControl(AirPlayReceiverSessionRef inSession, CFStringRef inCommand,
            CFTypeRef inQualifier, CFDictionaryRef inParams, CFDictionaryRef* outParams,
            void* inContext);
    static void HandleDuckAudio(AirPlayReceiverSessionRef inSession, double inDurationSecs,
            double inVolume, void* inContext);
    static void HandleUnduckAudio(AirPlayReceiverSessionRef inSession, double inDurationSecs,
            void* inContext);
    static CFTypeRef HandleCopyProperty(AirPlayReceiverSessionRef inSession, CFStringRef inProperty,
            CFTypeRef inQualifier, OSStatus* outErr, void* inContext);
    static OSStatus HandleSetProperty(AirPlayReceiverSessionRef inSession, CFStringRef inProperty,
            CFTypeRef inQualifier, CFTypeRef inValue, void* inContext);

    // command callbacks

    // IControlReceiver implementation
    void ChangeResourceMode(const ModeChanges& inChanges);
    static void ChangeResourceModeCallback(OSStatus inStatus, CFDictionaryRef inResponse,
            void *inContext);
    void RequestUI(const std::string& inUrl);
    void RequestSiriAction(SiriAction inAction);
    const std::string& GetVersion(VersionEntity entity);
    void SetBluetoothIDs(const std::list<std::string>& inDeviceIDs);
    void SetNightMode(bool inActive);
    void SetLimitedUI(bool inActive);
    void UpdateVehicleInformation(VehicleInformation inVehicleInfo);
    static void *iAP2WiFiCommThread(void* exinf);

    // private section
private:
    std::unique_ptr<IControlAdapter> controlAdapter;
    std::unique_ptr<InputChannel> inputChannel;
    Server* server;
    IDynamicConfiguration* config;
    std::string iOSBuildVersion;
    NightMode nightMode;
    bool limitedUI;
    bool supportLimitedUI;
    bool updateVehicleInfo;
    ETCInfo etcSupport;    // Electronic Toll Collection support
    NADInfo nadSupport;    // Navigation Aided Driving Support
    bool sessionInitialized;
    std::list<std::string> bluetoothIDs;
    pthread_mutex_t closeLock;
    bool closed;
    bool phoneSessionRunning;
    int iAP2OverCarPlay_fd;
    int sessionStop_fd;
    const uint64_t sessionStopEvent = 1;
    pthread_t iAP2WiFiCommThread_id;
    CarPlayTransportType Transport;
    std::mutex iAP2Mutex;
    std::condition_variable wiFiThreadCondVar;

    bool initControl();

    template<class T> std::unique_ptr<T> createAdapter(const std::string& inName);

    AudioChannel* audioChannel;
};

} } /* namespace adit { namespace carplay */

#endif /* CARPLAY_SESSION_H */
